﻿using inutralia.Models;
using MvvmHelpers; // Este namespace está en el paquete Refractored.MvvmHelpers
using System;
using System.Threading.Tasks;
using Xamarin.Forms;

namespace inutralia.ViewModels
{
    /// <summary>
    /// ViewModel del listado de preguntas. Hereda de MvvmHelpers.BaseViewModel
    /// </summary>
    public class QuestionListViewModel : BaseNavigationViewModel
    {
        public QuestionListViewModel() { }

        // Datos del histórico
        ObservableRangeCollection<QuestionSpecialist> _QuestionList;

        // Comando de loading del listado
        Command _LoadQuestionListCommand;

        // Comando de actualización del listado del listado
        Command _RefreshQuestionListCommand;

        // Acceso a la lista preguntas
        public ObservableRangeCollection<QuestionSpecialist> QuestionList
        {
            // Getter (lazy load), crea la lista si no exite
            get => _QuestionList ?? (_QuestionList = new ObservableRangeCollection<QuestionSpecialist>());

            // Setter. Cambia el valor y notifica a la vista de dicho cambio
            set
            {
                _QuestionList = value;
                OnPropertyChanged("QuestionList");
            }
        }

        public Command LoadQuestionListCommand
        {
            get { return _LoadQuestionListCommand ?? (_LoadQuestionListCommand = new Command(async () => await ExecuteLoadQuestionListCommand())); }
        }

        /// <summary>
        /// Método que realiza la carga inicial del listado
        /// </summary>
        /// 
        public async Task ExecuteLoadQuestionListCommand()
        {
            // Realiza el proceso de actualización si hay menos de un 
            // elemento en el listado
            if (QuestionList.Count < 1)
                await FetchQuestionList();
        }

        /// <summary>
        /// Acceso al comando de actualización del listado
        /// </summary>
        public Command RefreshQuestionListCommand
        {
            // Getter (lazy load), crea el comando si no existe
            get { return _RefreshQuestionListCommand ?? (_RefreshQuestionListCommand = new Command(async () => await ExecuteRefreshQuestionListCommand())); }
        }

        /// <summary>
        /// Proceso de ejecución del comando de actualización del listado
        /// </summary>
        async Task ExecuteRefreshQuestionListCommand()
        {
            try
            {
                // Hace que el comando no se pueda ejecutar de nuevo
                RefreshQuestionListCommand.ChangeCanExecute();

                // Realiza el proceso de actualización
                await FetchQuestionList();

                // Hace que el comando pueda volver a ejecutarse
                RefreshQuestionListCommand.ChangeCanExecute();
            }
            catch (Exception e)
            {
                string err = e.Message;
            }
        }

        /// <summary>
        /// Proceso de actualización del listado
        /// </summary>
        async Task FetchQuestionList()
        {
            // Indicamos que estamos ocupados (provoca que aparezca el indicador de carga)
            IsBusy = true;

            // Llamada al API para coger el listado (provoca que se actualize la vista del listado)
            // Nota: Al obtener el listado, el controlador Rest del servidor no retorna el cuerpo de
            // la notificación (campo Body)
            try
            {
                QuestionList = new ObservableRangeCollection<QuestionSpecialist>(await App.API.RefreshListAsync<QuestionSpecialist>());
            }
            catch (Exception e)
            {
                QuestionList.Clear();
                string err = e.Message;
            }

            // Indicamos que ya no estamos ocupados (provoca que desaparezca el indicador de carga)
            IsBusy = false;
        }
    }
}
