﻿namespace inutralia.ViewModels
{
    using inutralia.Models;
    using System.Threading.Tasks;
    using MvvmHelpers;
    using Xamarin.Forms;

    public class GenericListViewModel : BaseViewModel
    {
        public GenericListViewModel() { }

        //Lista de menús genéricos
        ObservableRangeCollection<Generic> _Generics;

        // Comando de update del listado
        Command _RefreshGenericsCommand;

        // Acceso a la lista de menús genéricos
        public ObservableRangeCollection<Generic> Generics
        {
            // Getter (lazy load), crea la lista si no existe
            get { return _Generics ?? (_Generics = new ObservableRangeCollection<Generic>()); }

            // Setter. Cambia el valor y notifica a la vista de dicho cambio
            set
            {
                _Generics = value;
                OnPropertyChanged("Generics");
            }
        }

        /// <summary>
        /// Método que realiza la carga inicial del listado
        /// </summary>
        public async Task ExecuteLoadGenericsCommand()
        {
            // Realiza el proceso de actualización si hay menos de un 
            // elemento en el listado
            if (Generics.Count < 1)
                await FetchGenerics();
        }

        /// <summary>
        /// Acceso al comando de actualización del listado
        /// </summary>
        public Command RefreshGenericsCommand
        {
            // Getter (lazy load), crea el comando si no existe
            get { return _RefreshGenericsCommand ?? (_RefreshGenericsCommand = new Command(async () => await ExecuteRefreshGenericsCommand())); }
        }


        /// <summary>
        /// Proceso de ejecución del comando de actualización del listado
        /// </summary>
        async Task ExecuteRefreshGenericsCommand()
        {
            // Hace que el comando no se pueda ejecutar de nuevo
            RefreshGenericsCommand.ChangeCanExecute();

            // Realiza el proceso de actualización
            await FetchGenerics();

            // Hace que el comando pueda volver a ejecutarse
            RefreshGenericsCommand.ChangeCanExecute();
        }

        /// <summary>
        /// Proceso de actualización del listado
        /// </summary>
        async Task FetchGenerics()
        {
            // Indicamos que estamos ocupados (provoca que aparezca el indicador de carga)
            IsBusy = true;

            // Llamada al API para coger el listado (provoca que se actualize la vista del listado)
            // Nota: Al obtener el listado, el controlador Rest del servidor no retorna el cuerpo de
            // la notificación (campo Body)
            try
            {
                Generics = new ObservableRangeCollection<Generic>(await App.API.RefreshListAsync<Generic>());
            }
            catch (System.Exception)
            {

                Generics.Clear();
            }

            // Indicamos que ya no estamos ocupados (provoca que desaparezca el indicador de carga)
            IsBusy = false;
        }
    }
}