﻿namespace inutralia.ViewModels
{
    using inutralia.Models;
    using inutralia.Views;
    using MvvmHelpers;
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Xamarin.Forms;

    public class TrivialListViewModel : BaseNavigationViewModel
    {
        public TrivialListViewModel() { }

        // Lista de partidas
        ObservableRangeCollection<TrivialGame> _Games;

        // Acceso a la lista de partidas
        public ObservableRangeCollection<TrivialGame> Games
        {
            // Getter (lazy load), crea la lista si no existe
            get { return _Games ?? (_Games = new ObservableRangeCollection<TrivialGame>()); }

            // Setter. cambiar el valor y notifica a la vista de dicho cambio
            set
            {
                _Games = value;
                OnPropertyChanged("Games");
            }
        }

        #region Crear partida

        Command _NewGameCommand;

        public Command NewGameCommand => _NewGameCommand ?? (_NewGameCommand = new Command(async () => await ExecuteNewGameCommand()));

        protected async Task ExecuteNewGameCommand()
        {
            NewGameCommand.ChangeCanExecute();

            List<TrivialQuestion> questions = null;

            try
            {
                questions = await App.API.RefreshListAsync<TrivialQuestion>();
            }
            catch (Exception e) { }

            if (questions == null) questions = new List<TrivialQuestion>()
                {
                    new TrivialQuestion ()
                    {
                        Id = -1,
                        Image = "",
                        Text = "¿Cuantas manzanas hay en la foto?",
                        Options = new List<string> () { "Ninguna", "Una", "Cuatro", "Más de cuatro" },
                        ValidIndex = 0
                    }
                };

            var game = TrivialGame.Create(questions);
            await App.LocalData.UpdateItemAsync(game, true);

            await PushAsync(new TrivialGameView()
            {
                BindingContext = new TrivialGameViewModel(game)
            });

            NewGameCommand.ChangeCanExecute();
        }

        #endregion

        #region Borrar partida

        Command _DeleteGameCommand;

        public Command DeleteGameCommand =>
            _DeleteGameCommand ?? (_DeleteGameCommand = new Command(async (parameter) => await ExecuteDeleteGameCommand(parameter as TrivialGame)));

        protected async Task ExecuteDeleteGameCommand(TrivialGame game)
        {
            // Verificar parámetro
            if (game != null)
            {
                // Indicamos que estamos ocupados (provoca que aparezca el indicador de carga)
                IsBusy = true;

                // Llamamos al API para borrar la notificación
                await App.LocalData.DeleteItemAsync(game);

                // Actualizamos la lista
                Games.Remove(game);

                // Indicamos que ya no estamos ocupados (provoca que desaparezca el indicador de carga)
                IsBusy = false;
            } //endif
        }

        #endregion


        /// <summary>
        /// Método que realiza la carga inicial del listado
        /// </summary>
        public async Task RefreshList()
        {
            await FetchGames();
        }

        /// <summary>
        /// Proceso de actualización del listado
        /// </summary>
        async Task FetchGames()
        {
            // Indicamos que estamos ocupados (provoca que aparezca el indicador de carga)
            IsBusy = true;

            // Llamada al API para coger el listado (provoca que se actualize la vista del listado)
            try
            {
                Games = new ObservableRangeCollection<TrivialGame>(await App.LocalData.RefreshListAsync<TrivialGame>());
            }
            catch (System.Exception e)
            {
                Games.Clear();
            }

            // Indicamos que ya no estamos ocupados (provoca que desaparezca el indicador de carga)
            IsBusy = false;
        }
    }
}
